<?php
$sessionStatus = session_status();
if ($sessionStatus === PHP_SESSION_NONE) {
    session_start();
}

date_default_timezone_set('America/Santiago');

$autoload = __DIR__ . '/../vendor/autoload.php';
if (!file_exists($autoload)) {
    http_response_code(500);
    echo "Falta vendor/autoload.php. Ejecuta composer para instalar Slim.";
    exit;
}

require $autoload;

use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;
use Slim\Factory\AppFactory;

loadEnv(__DIR__ . '/../.env');

$pdo = buildPdoFromEnv();

$app = AppFactory::create();
$basePath = env('APP_BASE_PATH', null);
if ($basePath === null) {
    $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
    $basePath = rtrim(str_replace('/index.php', '', $scriptName), '/');
}
if ($basePath !== '') {
    $app->setBasePath($basePath);
}
$app->addErrorMiddleware(true, true, true);

$app->get('/', function (Request $request, Response $response) use ($pdo, $basePath) {
    $query = $request->getQueryParams();
    $step = $query['step'] ?? 'welcome';
    $toast = consumeFlash('toast');
    $errors = consumeFlash('errors') ?? [];
    $old = consumeFlash('old') ?? [];

    $panoramaError = ($query['err'] ?? '') === 'panorama' || !empty($_SESSION['panorama_error']);
    $panorama = $_SESSION['panorama'] ?? [];

    if (!empty($_SESSION['panorama_error'])) {
        unset($_SESSION['panorama_error']);
    }

    if ($panoramaError) {
        $toast = $toast ?: [
            'type' => 'err',
            'title' => 'Caso no evaluable',
            'message' => 'Seleccionaste al menos un “Sí”. Te mostramos ayuda inmediata.',
        ];
    }

    if ($step === 'imagenes' && empty($_SESSION['datos'])) {
        setFlash('toast', [
            'type' => 'warn',
            'title' => 'Acceso inválido',
            'message' => 'Primero debes completar tus datos.',
        ]);
        return redirect($response, '/?step=datos');
    }

    if ($step === 'loading' && empty($_SESSION['pending_analysis'])) {
        setFlash('toast', [
            'type' => 'warn',
            'title' => 'Sin análisis pendiente',
            'message' => 'No hay imágenes en proceso.',
        ]);
        return redirect($response, '/?step=imagenes');
    }

    if ($step === 'informe') {
        $informe = $_SESSION['informe'] ?? null;
        if (!$informe && !empty($_SESSION['solicitud_id'])) {
            $informe = fetchInforme($pdo, (int)$_SESSION['solicitud_id']);
            if ($informe) {
                $_SESSION['informe'] = $informe;
            }
        }
    } else {
        $informe = null;
    }

    $attemptInfo = null;
    if ($step === 'imagenes' && !empty($_SESSION['datos']['solicitud_id'])) {
        $attempts = countAttempts($pdo, (int)$_SESSION['datos']['solicitud_id']);
        $remaining = max(0, 3 - $attempts);
        if ($remaining <= 0) {
            setFlash('result_neg', [
                'message' => 'Ya alcanzaste el máximo de 3 intentos hoy.',
                'remaining' => 0,
            ]);
            return redirect($response, '/?step=result-neg');
        }
        $attemptInfo = [
            'attempts' => $attempts,
            'remaining' => $remaining,
        ];
    }

    $resultNeg = consumeFlash('result_neg') ?? ($_SESSION['last_result_neg'] ?? null);
    $resultPos = consumeFlash('result_pos') ?? ($_SESSION['last_result_pos'] ?? null);

    ob_start();
    $currentStep = $step;
    $initialStep = $step;
    $panoramaError = $panoramaError;
    $panorama = $panorama;
    $toast = $toast;
    $errors = $errors;
    $old = $old;
    $datos = $_SESSION['datos'] ?? [];
    $attemptInfo = $attemptInfo;
    $assetBase = $basePath !== '' ? $basePath : '';
    $fechaIngreso = date('Y-m-d H:i');
    $resultNeg = $resultNeg;
    $resultPos = $resultPos;
    $informe = $informe;
    include __DIR__ . '/../views/app.php';
    $html = ob_get_clean();
    $response->getBody()->write($html);
    return $response;
});

$app->post('/panorama', function (Request $request, Response $response) {
    $data = (array)$request->getParsedBody();
    $keys = ['injury', 'fire', 'engine', 'fuel'];
    $hasYes = false;
    foreach ($keys as $key) {
        if (($data[$key] ?? 'no') === 'si') {
            $hasYes = true;
            break;
        }
    }

    $_SESSION['panorama'] = [
        'injury' => $data['injury'] ?? 'no',
        'fire' => $data['fire'] ?? 'no',
        'engine' => $data['engine'] ?? 'no',
        'fuel' => $data['fuel'] ?? 'no',
    ];

    if ($hasYes) {
        $_SESSION['panorama_error'] = true;
        return redirect($response, '/?step=panorama&err=panorama');
    }

    return redirect($response, '/?step=datos');
});

$app->post('/datos', function (Request $request, Response $response) use ($pdo) {
    $data = (array)$request->getParsedBody();
    $name = trim((string)($data['name'] ?? ''));
    $rutRaw = trim((string)($data['rut'] ?? ''));
    $plateRaw = trim((string)($data['plate'] ?? ''));
    $email = trim((string)($data['email'] ?? ''));

    $errors = [];
    if ($name === '') $errors['name'] = 'El nombre es obligatorio.';
    if ($rutRaw === '' || !isValidRut($rutRaw)) $errors['rut'] = 'RUT inválido.';
    if ($plateRaw === '' || !isValidPlate($plateRaw)) $errors['plate'] = 'Patente inválida.';
    if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors['email'] = 'Email inválido.';

    if (!empty($errors)) {
        setFlash('errors', $errors);
        setFlash('old', [
            'name' => $name,
            'rut' => $rutRaw,
            'plate' => $plateRaw,
            'email' => $email,
        ]);
        setFlash('toast', [
            'type' => 'err',
            'title' => 'Revisa el formulario',
            'message' => 'Corrige los campos marcados para continuar.',
        ]);
        return redirect($response, '/?step=datos');
    }

    $rut = normalizeRut($rutRaw);
    $plate = normalizePlate($plateRaw);
    $today = date('Y-m-d');

    $existing = findSolicitudToday($pdo, $rut, $plate, $today);
    if ($existing) {
        $_SESSION['solicitud_id'] = (int)$existing['id_solicitud'];
        if (hasPaidReport($pdo, (int)$existing['id_solicitud'])) {
            $informe = fetchInforme($pdo, (int)$existing['id_solicitud']);
            if ($informe) {
                $_SESSION['informe'] = $informe;
                setFlash('toast', [
                    'type' => 'ok',
                    'title' => 'Informe disponible',
                    'message' => 'Ya existe un informe vigente para hoy.',
                ]);
                return redirect($response, '/?step=informe');
            }
        }
    }

    if (!$existing) {
        $stmt = $pdo->prepare('INSERT INTO solicitud (nombre, rut, patente, email, fecha, cantidad_img) VALUES (:nombre, :rut, :patente, :email, NOW(), 0)');
        $stmt->execute([
            ':nombre' => $name,
            ':rut' => $rut,
            ':patente' => $plate,
            ':email' => $email,
        ]);
        $solicitudId = (int)$pdo->lastInsertId();
    } else {
        $solicitudId = (int)$existing['id_solicitud'];
    }

    $_SESSION['datos'] = [
        'solicitud_id' => $solicitudId,
        'name' => $name,
        'rut' => $rut,
        'plate' => $plate,
        'email' => $email,
        'fecha' => date('Y-m-d H:i'),
    ];
    $_SESSION['solicitud_id'] = $solicitudId;
    $_SESSION['form_token'] = bin2hex(random_bytes(16));
    $_SESSION['informe'] = null;

    return redirect($response, '/?step=imagenes');
});

$app->post('/imagenes', function (Request $request, Response $response) use ($pdo) {
    if (empty($_SESSION['datos'])) {
        setFlash('toast', [
            'type' => 'warn',
            'title' => 'Acceso inválido',
            'message' => 'Primero debes completar tus datos.',
        ]);
        return redirect($response, '/?step=datos');
    }

    $data = (array)$request->getParsedBody();
    $token = (string)($data['token'] ?? '');
    if (empty($_SESSION['form_token']) || !hash_equals($_SESSION['form_token'], $token)) {
        setFlash('toast', [
            'type' => 'err',
            'title' => 'Solicitud inválida',
            'message' => 'Los datos no provienen del formulario de datos.',
        ]);
        return redirect($response, '/?step=datos');
    }

    $solicitudId = (int)($_SESSION['datos']['solicitud_id'] ?? 0);
    if ($solicitudId <= 0) {
        setFlash('toast', [
            'type' => 'err',
            'title' => 'Solicitud inválida',
            'message' => 'No se encontró una solicitud válida.',
        ]);
        return redirect($response, '/?step=datos');
    }

    $attempts = countAttempts($pdo, $solicitudId);
    if ($attempts >= 3) {
        setFlash('result_neg', [
            'message' => 'Ya alcanzaste el máximo de 3 intentos hoy.',
            'remaining' => 0,
        ]);
        return redirect($response, '/?step=result-neg');
    }

    $files = $request->getUploadedFiles();
    $images = $files['images'] ?? [];
    if (!is_array($images)) $images = [$images];
    $images = array_values(array_filter($images, fn($f) => $f->getError() !== UPLOAD_ERR_NO_FILE));

    if (count($images) < 3 || count($images) > 6) {
        setFlash('toast', [
            'type' => 'err',
            'title' => 'Imágenes inválidas',
            'message' => 'Debes cargar entre 3 y 6 imágenes.',
        ]);
        return redirect($response, '/?step=imagenes');
    }

    $allowedMime = ['image/jpeg' => 'jpg', 'image/png' => 'png'];
    $storedPaths = [];
    $storageDir = __DIR__ . '/../storage/uploads';
    if (!is_dir($storageDir)) {
        mkdir($storageDir, 0775, true);
    }

    foreach ($images as $img) {
        if ($img->getError() !== UPLOAD_ERR_OK) {
            setFlash('toast', [
                'type' => 'err',
                'title' => 'Error de subida',
                'message' => 'No pudimos leer una de las imágenes.',
            ]);
            return redirect($response, '/?step=imagenes');
        }
        $mime = $img->getClientMediaType();
        if (!isset($allowedMime[$mime])) {
            setFlash('toast', [
                'type' => 'err',
                'title' => 'Formato inválido',
                'message' => 'Solo se permiten archivos PNG o JPG.',
            ]);
            return redirect($response, '/?step=imagenes');
        }
    }

    $nextIndex = nextImageIndex($pdo, $solicitudId);
    foreach ($images as $img) {
        $mime = $img->getClientMediaType();
        $ext = $allowedMime[$mime];
        $fileName = $solicitudId . '_' . $nextIndex . '.' . $ext;
        $target = $storageDir . '/' . $fileName;
        $img->moveTo($target);
        $storedPaths[] = $target;

        $stmt = $pdo->prepare('INSERT INTO imagen (id_solicitud, path) VALUES (:id, :path)');
        $stmt->execute([
            ':id' => $solicitudId,
            ':path' => $target,
        ]);
        $nextIndex++;
    }

    $stmt = $pdo->prepare('UPDATE solicitud SET cantidad_img = (SELECT COUNT(*) FROM imagen WHERE id_solicitud = :id) WHERE id_solicitud = :id');
    $stmt->execute([':id' => $solicitudId]);

    $_SESSION['pending_analysis'] = [
        'solicitud_id' => $solicitudId,
        'files' => $storedPaths,
        'attempt' => $attempts + 1,
    ];

    return redirect($response, '/?step=loading');
});

$app->post('/procesar', function (Request $request, Response $response) use ($pdo) {
    if (empty($_SESSION['pending_analysis'])) {
        $payload = json_encode(['status' => 'error', 'message' => 'No hay análisis pendiente.']);
        $response->getBody()->write($payload);
        return $response->withHeader('Content-Type', 'application/json')->withStatus(400);
    }

    $pending = $_SESSION['pending_analysis'];
    unset($_SESSION['pending_analysis']);

    $solicitudId = (int)$pending['solicitud_id'];
    $attempt = (int)$pending['attempt'];
    $files = $pending['files'] ?? [];

    $analysis = analyzeImages($files);
    $statusRaw = $analysis['status'] ?? 'fail';
    $isInternalError = !in_array($statusRaw, ['success', 'fail'], true);
    $status = $isInternalError ? 'fail' : $statusRaw;
    $tipoAuto = $analysis['tipo_auto'] ?? 'No informado';
    $costo = (float)($analysis['costo_total'] ?? 0);
    $descripcion = $analysis['descripcion'] ?? ($analysis['detalle'] ?? 'Sin detalle');

    $stmt = $pdo->prepare('INSERT INTO respuesta_solicitud (id_solicitud, estado, detalle, intento) VALUES (:id, :estado, :detalle, :intento)');
    $stmt->execute([
        ':id' => $solicitudId,
        ':estado' => $status,
        ':detalle' => $descripcion,
        ':intento' => $attempt,
    ]);
    $respuestaId = (int)$pdo->lastInsertId();

    if ($status === 'success') {
        $stmt = $pdo->prepare('INSERT INTO solicitud_aprobada (id_respuesta_solicitud, tipo_auto, costo_total, descripcion) VALUES (:id, :tipo, :costo, :descripcion)');
        $stmt->execute([
            ':id' => $respuestaId,
            ':tipo' => $tipoAuto,
            ':costo' => $costo,
            ':descripcion' => $descripcion,
        ]);

        $payloadData = [
            'solicitud_id' => $solicitudId,
            'tipo_auto' => $tipoAuto,
            'costo_total' => $costo,
            'descripcion' => $descripcion,
        ];
        $_SESSION['last_result_pos'] = $payloadData;
        setFlash('result_pos', $payloadData);
        $payload = json_encode(['status' => 'success', 'redirect' => '/?step=result-pos']);
        $response->getBody()->write($payload);
        return $response->withHeader('Content-Type', 'application/json');
    }

    $remaining = max(0, 3 - $attempt);
    $negData = [
        'message' => $isInternalError ? 'Error interno en el análisis.' : 'No fue posible evaluar con claridad.',
        'remaining' => $remaining,
    ];
    $_SESSION['last_result_neg'] = $negData;
    setFlash('result_neg', $negData);
    $payload = json_encode(['status' => 'fail', 'redirect' => '/?step=result-neg', 'remaining' => $remaining]);
    $response->getBody()->write($payload);
    return $response->withHeader('Content-Type', 'application/json');
});

$app->post('/pago', function (Request $request, Response $response) use ($pdo) {
    $data = (array)$request->getParsedBody();
    $result = (string)($data['result'] ?? '');
    $solicitudId = (int)($data['solicitud_id'] ?? 0);
    if ($solicitudId <= 0) {
        setFlash('toast', [
            'type' => 'err',
            'title' => 'Pago inválido',
            'message' => 'No se encontró la solicitud.',
        ]);
        return redirect($response, '/?step=result-pos');
    }

    if ($result !== 'success') {
        setFlash('toast', [
            'type' => 'err',
            'title' => 'Pago rechazado',
            'message' => 'No pudimos confirmar el pago. Intenta nuevamente.',
        ]);
        return redirect($response, '/?step=result-pos');
    }

    if (hasPaidReport($pdo, $solicitudId)) {
        $informe = fetchInforme($pdo, $solicitudId);
        if ($informe) {
            $_SESSION['informe'] = $informe;
        }
        setFlash('toast', [
            'type' => 'ok',
            'title' => 'Pago ya registrado',
            'message' => 'Tu informe ya está disponible.',
        ]);
        return redirect($response, '/?step=informe');
    }

    $aprobada = fetchAprobada($pdo, $solicitudId);
    $monto = $aprobada ? (float)$aprobada['costo_total'] : 0;

    $stmt = $pdo->prepare('INSERT INTO pago (id_solicitud, monto, moneda, gateway, mp_payment_id, mp_preference_id, fecha_confirmacion) VALUES (:id, :monto, :moneda, :gateway, :mp_payment, :mp_preference, NOW())');
    $stmt->execute([
        ':id' => $solicitudId,
        ':monto' => $monto,
        ':moneda' => 'CLP',
        ':gateway' => 'mercadopago-demo',
        ':mp_payment' => uniqid('demo_', true),
        ':mp_preference' => uniqid('pref_', true),
    ]);

    $informe = fetchInforme($pdo, $solicitudId);
    if ($informe) {
        $_SESSION['informe'] = $informe;
    }

    setFlash('toast', [
        'type' => 'ok',
        'title' => 'Pago confirmado',
        'message' => 'Puedes descargar tu informe.',
    ]);
    return redirect($response, '/?step=informe');
});

$app->run();

function loadEnv(string $path): void
{
    if (!file_exists($path)) {
        return;
    }
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || str_starts_with($line, '#')) {
            continue;
        }
        [$key, $value] = array_pad(explode('=', $line, 2), 2, '');
        $key = trim($key);
        $value = trim($value);
        $value = trim($value, "\"'");
        if ($key !== '') {
            $_ENV[$key] = $value;
        }
    }
}

function env(string $key, ?string $default = null): ?string
{
    return $_ENV[$key] ?? $default;
}

function buildPdoFromEnv(): PDO
{
    $host = env('DB_HOST', '127.0.0.1');
    $port = env('DB_PORT', '3306');
    $db = env('DB_NAME', 'choque');
    $user = env('DB_USER', 'root');
    $pass = env('DB_PASS', '');
    $charset = env('DB_CHARSET', 'utf8mb4');
    $dsn = "mysql:host={$host};port={$port};dbname={$db};charset={$charset}";
    $pdo = new PDO($dsn, $user, $pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
    return $pdo;
}

function redirect(Response $response, string $url): Response
{
    return $response->withHeader('Location', $url)->withStatus(302);
}

function setFlash(string $key, $value): void
{
    $_SESSION['flash'][$key] = $value;
}

function consumeFlash(string $key)
{
    if (!isset($_SESSION['flash'][$key])) return null;
    $value = $_SESSION['flash'][$key];
    unset($_SESSION['flash'][$key]);
    return $value;
}

function normalizeRut(string $rut): string
{
    $r = strtoupper(str_replace(['.', ' '], '', $rut));
    if (!str_contains($r, '-')) {
        $body = substr($r, 0, -1);
        $dv = substr($r, -1);
        return $body . '-' . $dv;
    }
    return $r;
}

function formatRut(string $rut): string
{
    $n = normalizeRut($rut);
    $parts = explode('-', $n);
    if (count($parts) !== 2) return $rut;
    $body = preg_replace('/\D/', '', $parts[0]);
    $dv = strtoupper($parts[1]);
    if ($body === '' || $dv === '') return $rut;
    $out = '';
    $rev = str_split(strrev($body));
    foreach ($rev as $i => $digit) {
        $out = $digit . $out;
        if (($i + 1) % 3 === 0 && ($i + 1) !== count($rev)) {
            $out = '.' . $out;
        }
    }
    return $out . '-' . $dv;
}

function isValidRut(string $rut): bool
{
    $n = normalizeRut($rut);
    $parts = explode('-', $n);
    if (count($parts) !== 2) return false;
    $body = preg_replace('/\D/', '', $parts[0]);
    $dv = strtoupper($parts[1]);
    if (strlen($body) < 7 || strlen($body) > 8) return false;
    if (!preg_match('/^\d+$/', $body)) return false;
    if (!preg_match('/^[0-9K]$/', $dv)) return false;
    return rutDv($body) === $dv;
}

function rutDv(string $body): string
{
    $sum = 0;
    $mul = 2;
    for ($i = strlen($body) - 1; $i >= 0; $i--) {
        $sum += intval($body[$i]) * $mul;
        $mul = ($mul === 7) ? 2 : $mul + 1;
    }
    $mod = 11 - ($sum % 11);
    if ($mod === 11) return '0';
    if ($mod === 10) return 'K';
    return (string)$mod;
}

function normalizePlate(string $p): string
{
    return strtoupper(preg_replace('/[\s\-]/', '', $p));
}

function isValidPlate(string $p): bool
{
    $x = normalizePlate($p);
    return preg_match('/^[A-Z]{4}\d{2}$/', $x) || preg_match('/^[A-Z]{2}\d{4}$/', $x);
}

function findSolicitudToday(PDO $pdo, string $rut, string $plate, string $today): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM solicitud WHERE rut = :rut AND patente = :patente AND DATE(fecha) = :today ORDER BY id_solicitud DESC LIMIT 1');
    $stmt->execute([
        ':rut' => $rut,
        ':patente' => $plate,
        ':today' => $today,
    ]);
    $row = $stmt->fetch();
    return $row ?: null;
}

function hasPaidReport(PDO $pdo, int $solicitudId): bool
{
    $stmt = $pdo->prepare('SELECT id_pago FROM pago WHERE id_solicitud = :id LIMIT 1');
    $stmt->execute([':id' => $solicitudId]);
    return (bool)$stmt->fetch();
}

function fetchAprobada(PDO $pdo, int $solicitudId): ?array
{
    $stmt = $pdo->prepare('
        SELECT a.*
        FROM solicitud_aprobada a
        INNER JOIN respuesta_solicitud r ON r.id_respuesta_solicitud = a.id_respuesta_solicitud
        WHERE r.id_solicitud = :id
        ORDER BY a.id_solcitud_aprobada DESC
        LIMIT 1
    ');
    $stmt->execute([':id' => $solicitudId]);
    $row = $stmt->fetch();
    return $row ?: null;
}

function fetchInforme(PDO $pdo, int $solicitudId): ?array
{
    $stmt = $pdo->prepare('
        SELECT s.id_solicitud, s.nombre, s.rut, s.patente, s.email, s.fecha,
               a.tipo_auto, a.costo_total, a.descripcion
        FROM solicitud s
        INNER JOIN respuesta_solicitud r ON r.id_solicitud = s.id_solicitud
        INNER JOIN solicitud_aprobada a ON a.id_respuesta_solicitud = r.id_respuesta_solicitud
        WHERE s.id_solicitud = :id
        ORDER BY r.id_respuesta_solicitud DESC
        LIMIT 1
    ');
    $stmt->execute([':id' => $solicitudId]);
    $row = $stmt->fetch();
    return $row ?: null;
}

function countAttempts(PDO $pdo, int $solicitudId): int
{
    $stmt = $pdo->prepare('SELECT COUNT(*) AS total FROM respuesta_solicitud WHERE id_solicitud = :id');
    $stmt->execute([':id' => $solicitudId]);
    $row = $stmt->fetch();
    return $row ? (int)$row['total'] : 0;
}

function nextImageIndex(PDO $pdo, int $solicitudId): int
{
    $stmt = $pdo->prepare('SELECT COUNT(*) AS total FROM imagen WHERE id_solicitud = :id');
    $stmt->execute([':id' => $solicitudId]);
    $row = $stmt->fetch();
    return $row ? ((int)$row['total'] + 1) : 1;
}

function analyzeImages(array $paths): array
{
    $ch = curl_init();
    $url = 'https://127.0.0.1:5000/analyze';
    $post = [];
    foreach ($paths as $i => $path) {
        $mime = mime_content_type($path) ?: 'image/jpeg';
        $post["files[$i]"] = new CURLFile($path, $mime, basename($path));
    }
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $post,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HEADER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
    ]);

    $raw = curl_exec($ch);
    if ($raw === false) {
        curl_close($ch);
        return ['status' => 'fail', 'descripcion' => 'Error interno al analizar imágenes'];
    }

    $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
    $headers = substr($raw, 0, $headerSize);
    $body = substr($raw, $headerSize);
    curl_close($ch);

    $json = extractJsonFromHeaders($headers);
    if (!$json) {
        $json = extractJsonFromBody($body);
    }
    if (!$json) {
        return ['status' => 'fail', 'descripcion' => 'Error interno en respuesta de análisis'];
    }
    return $json;
}

function extractJsonFromHeaders(string $headers): ?array
{
    $lines = explode("\r\n", $headers);
    foreach ($lines as $line) {
        $parts = explode(':', $line, 2);
        if (count($parts) !== 2) continue;
        $value = trim($parts[1]);
        $decoded = tryDecodeJson($value);
        if ($decoded) return normalizeAnalysisJson($decoded);
    }
    return null;
}

function extractJsonFromBody(string $body): ?array
{
    $body = trim($body);
    if ($body === '') return null;
    $decoded = tryDecodeJson($body);
    return $decoded ? normalizeAnalysisJson($decoded) : null;
}

function tryDecodeJson(string $value): ?array
{
    if ($value === '') return null;
    $value = trim($value);
    if ($value[0] !== '{' || !str_ends_with($value, '}')) {
        return null;
    }
    $decoded = json_decode($value, true);
    return is_array($decoded) ? $decoded : null;
}

function normalizeAnalysisJson(array $data): array
{
    $status = $data['status'] ?? $data['estado'] ?? null;
    $status = $status ? strtolower((string)$status) : null;
    if ($status === 'ok' || $status === 'approved') $status = 'success';
    if ($status === 'error' || $status === 'failed') $status = 'fail';

    return [
        'tipo_auto' => $data['tipo_auto'] ?? $data['tipo'] ?? $data['tipoAuto'] ?? null,
        'costo_total' => $data['costo_total'] ?? $data['costo'] ?? $data['costoTotal'] ?? 0,
        'descripcion' => $data['descripcion'] ?? $data['detalle'] ?? $data['Descripcion Detallada'] ?? null,
        'status' => $status ?? 'fail',
    ];
}
