// ===== PWA inline manifest + icon (data URI) =====
(function setupPwaInline() {
  const svgIcon = `
<svg xmlns="http://www.w3.org/2000/svg" width="512" height="512">
  <defs>
    <linearGradient id="g" x1="0" y1="0" x2="1" y2="1">
      <stop offset="0" stop-color="#2563eb"/>
      <stop offset="1" stop-color="#1d4ed8"/>
    </linearGradient>
  </defs>
  <rect width="512" height="512" rx="112" fill="url(#g)"/>
  <text x="50%" y="54%" text-anchor="middle" font-family="Arial, Helvetica, sans-serif"
        font-size="210" fill="white">C</text>
</svg>`.trim();

  const iconDataUri = "data:image/svg+xml;charset=utf-8," + encodeURIComponent(svgIcon);
  const manifest = {
    name: "Choque.cl",
    short_name: "Choque.cl",
    start_url: "./?step=welcome",
    scope: "./",
    display: "standalone",
    background_color: "#0f172a",
    theme_color: "#2563eb",
    icons: [
      { src: iconDataUri, sizes: "192x192", type: "image/svg+xml", purpose: "any" },
      { src: iconDataUri, sizes: "512x512", type: "image/svg+xml", purpose: "any" }
    ]
  };

  const manifestDataUri = "data:application/manifest+json;charset=utf-8," + encodeURIComponent(JSON.stringify(manifest));
  const manifestLink = document.getElementById("manifestLink");
  const appleTouch = document.getElementById("appleTouchIcon");
  if (manifestLink) manifestLink.setAttribute("href", manifestDataUri);
  if (appleTouch) appleTouch.setAttribute("href", iconDataUri);
})();

// ===== Modal helpers =====
(function initModals() {
  const modalHelp = document.getElementById("modalHelp");
  document.querySelectorAll("[data-open-modal]").forEach(btn => {
    btn.addEventListener("click", () => {
      const which = btn.getAttribute("data-open-modal");
      if (which === "help" && modalHelp) {
        modalHelp.classList.remove("hidden");
        modalHelp.classList.add("flex");
      }
    });
  });
  document.querySelectorAll("[data-close-modal]").forEach(btn => {
    btn.addEventListener("click", () => {
      const which = btn.getAttribute("data-close-modal");
      if (which === "help" && modalHelp) {
        modalHelp.classList.add("hidden");
        modalHelp.classList.remove("flex");
      }
    });
  });
  if (modalHelp) {
    modalHelp.addEventListener("click", (e) => {
      if (e.target === modalHelp) {
        modalHelp.classList.add("hidden");
        modalHelp.classList.remove("flex");
      }
    });
  }
})();

// ===== Image previews + dynamic inputs =====
(function initImages() {
  const addBtn = document.getElementById("btnAddImage");
  const extraContainer = document.getElementById("extraImages");

  function totalInputs() {
    return document.querySelectorAll(".image-input").length;
  }

  function addImageField() {
    const count = totalInputs();
    if (count >= 6) return;
    const index = count + 1;
    const card = document.createElement("div");
    card.className = "rounded-2xl border p-4 mt-4";
    card.innerHTML = `
      <div class="flex items-center justify-between gap-3">
        <div>
          <div class="text-sm font-semibold">Foto ${index} – Complementaria</div>
          <div class="text-xs text-slate-500">Agrega contexto adicional si lo necesitas.</div>
        </div>
        <label class="inline-flex cursor-pointer items-center justify-center rounded-xl bg-blue-600 px-4 py-2.5 text-sm font-semibold text-white hover:bg-blue-700">
          📷 Tomar foto
          <input name="images[]" type="file" accept="image/png,image/jpeg" capture="environment" class="hidden image-input" required />
        </label>
      </div>
      <div class="mt-4 flex h-40 items-center justify-center rounded-xl border border-dashed bg-slate-50 text-sm text-slate-500 image-preview">
        Vista previa
      </div>
    `;
    if (extraContainer) extraContainer.appendChild(card);
    if (totalInputs() >= 6 && addBtn) addBtn.disabled = true;
  }

  if (addBtn) {
    addBtn.addEventListener("click", () => addImageField());
  }

  document.addEventListener("change", (e) => {
    const input = e.target;
    if (!(input instanceof HTMLInputElement)) return;
    if (!input.classList.contains("image-input")) return;

    const file = input.files && input.files[0];
    if (!file) return;

    const card = input.closest(".rounded-2xl");
    const preview = card ? card.querySelector(".image-preview") : null;
    if (!preview) return;

    const url = URL.createObjectURL(file);
    preview.className = "mt-4 h-40 overflow-hidden rounded-xl border bg-white image-preview";
    preview.innerHTML = `<img src="${url}" alt="preview" class="h-full w-full object-cover" />`;
  });
})();

// ===== Loading + processing =====
(function initLoading() {
  const view = document.getElementById("view-loading");
  if (!view || view.classList.contains("hidden")) return;

  const base = document.body?.dataset?.base || "";
  const bar = document.getElementById("fakeBar");
  const pct = document.getElementById("fakePct");

  function setProgress(val, text) {
    if (bar) bar.style.width = val + "%";
    if (pct) pct.textContent = text;
  }

  setProgress(15, "Enviando imágenes…");
  const tick = setInterval(() => {
    const current = parseInt(bar?.style.width || "15", 10);
    if (current < 85) setProgress(current + 5, "Analizando imágenes…");
  }, 900);

  fetch(base + "/procesar", { method: "POST" })
    .then(res => res.json())
    .then(data => {
      clearInterval(tick);
      setProgress(100, "Finalizando…");
      setTimeout(() => {
        if (data && data.redirect) {
          window.location.href = data.redirect;
        }
      }, 500);
    })
    .catch(() => {
      clearInterval(tick);
      setProgress(100, "Error interno al analizar");
    });
})();

// ===== Demo download =====
(function initDownload() {
  const btn = document.getElementById("btnDownload");
  if (!btn) return;

  btn.addEventListener("click", () => {
    const content = [
      "CHOQUE.CL - INFORME (DEMO)",
      "===========================",
      "",
      `Fecha: ${new Date().toLocaleString("es-CL")}`,
      "",
      "Este archivo es un demo generado localmente.",
    ].join("\n");

    const blob = new Blob([content], { type: "application/pdf" });
    const url = URL.createObjectURL(blob);

    const a = document.createElement("a");
    a.href = url;
    a.download = "Informe_Choque_DEMO.pdf";
    document.body.appendChild(a);
    a.click();
    a.remove();
    setTimeout(() => URL.revokeObjectURL(url), 2000);
  });
})();
